/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx__fops.c,v 1.37 2006/05/11 22:26:29 eugene Exp $";

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx_io.h"
#include "mx__fops.h"
#include "mx__valgrind.h"

#ifdef MX_KERNEL
#error This file should not be compiled in the kernel
#endif

#if MX_OS_LINUX
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <errno.h>
#elif MX_OS_WINNT
#include <windows.h>
#elif MX_OS_FREEBSD || MX_OS_MACOSX
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#elif MX_OS_SOLARIS
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stropts.h>
#include <sys/mman.h>
#include <sys/errno.h>
#elif MX_OS_UDRV
#include <sys/socket.h>
#include <sys/un.h>
#include <alloca.h>
#include <unistd.h>
#include <sys/mman.h>
#include "mx__udrv_fops.h"
#else
#error
#endif

#include "mx__lib_types.h"
#include "mx_io_impl.h"
#include "mx__internals.h"

/* For systems w/o cloning devices, we use the convenction that:
   endpoint == -1 means open the generic board device  (/dev/mxp0)
   endpoint == -2 means open the raw board device  (/dev/mxr0) */

MX_FUNC(mx_return_t)
mx__open(int unit, int endpoint, mx_endpt_handle_t *handle)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS
  char buff[80];
  int rc;
  extern int errno;

  if (!Mx_init_count) {
    return MX_NOT_INITIALIZED;
  }
  mx_sprintf(buff, "/dev/mxp%d", unit);

  if (MX_OS_MACOSX) {
    switch (endpoint) {
    case -2: 
      mx_sprintf(buff, "/dev/mxr%d", unit); break;
    case -1: 
      mx_sprintf(buff, "/dev/mxp%d", unit); break;
    default: 
      mx_sprintf(buff, "/dev/mx%de%d", unit, endpoint); break;
    }
  }

  rc = open(buff, O_RDWR);

  if (rc != -1) {
    *handle = rc;
    return MX_SUCCESS;
  } 

  /* try again with unprivledged devices */

  if (MX_OS_MACOSX && endpoint >= 0) {
    mx_sprintf(buff, "/dev/mx%de%d", unit, endpoint);
  } else {
    mx_sprintf(buff, "/dev/mx%d", unit);
  }

  rc = open(buff, O_RDWR);
  
  if (rc != -1) {
    *handle = rc;
    return MX_SUCCESS;
  }	

  if (errno == ENOENT) {
    return MX_NO_DEV;
  } else if (errno == ENODEV || errno == ENXIO) {
    return MX_NO_DRIVER;
  } else if (errno == EACCES) {
    return MX_NO_PERM;
  } else if (errno == EBUSY) {
    return MX_BUSY;
  } else {
    return MX_NO_RESOURCES;
  }

#elif MX_OS_WINNT
  char buff[80];

  /* TODO: Error handling. */
  if (endpoint == -1) {
    mx_sprintf(buff, "\\\\.\\mx%d\\p", unit);
    *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			 OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (*handle != INVALID_HANDLE_VALUE) {
      return MX_SUCCESS;
    }
    else {
      mx_sprintf(buff, "\\\\.\\mx%d", unit);
      *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			   OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
      if (*handle != INVALID_HANDLE_VALUE) {
	return MX_SUCCESS;
      }
      else {
	return MX_NO_RESOURCES;
      }
    }
  }
  else if (endpoint == -2) {
    mx_sprintf(buff, "\\\\.\\mx%d\\r", unit);
    *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			 OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (*handle != INVALID_HANDLE_VALUE) {
      return MX_SUCCESS;
    }
    else {
      return MX_NO_RESOURCES;
    }
  }
  else {
    mx_sprintf(buff, "\\\\.\\mx%d\\e%d", unit, endpoint);
    *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			 OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (*handle != INVALID_HANDLE_VALUE) {
      return MX_SUCCESS;
    }
    else {
      return MX_NO_RESOURCES;
    }
  }
#elif MX_OS_UDRV
  return mx__udrv_open(unit, endpoint, handle);
#else
#error
#endif
}

/* Try to open /dev/mxctlp, then /dev/mxctl. */
MX_FUNC(mx_return_t)
mx_open_ctl(int priv, mx_endpt_handle_t *handle)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS
  char buff[80];
  int rc;
  extern int errno;

  if (!Mx_init_count) {
    return MX_NOT_INITIALIZED;
  }
  strcpy(buff, "/dev/mxctlp");

  rc = open(buff, O_RDWR);

  if (rc != -1) {
    *handle = rc;
    return MX_SUCCESS;
  } 

  /* try again with unprivledged devices */

  strcpy(buff, "/dev/mxctl");

  rc = open(buff, O_RDWR);
  
  if (rc != -1) {
    *handle = rc;
    return MX_SUCCESS;
  }	

  if (errno == ENOENT) {
    return MX_NO_DEV;
  } else if (errno == ENODEV || errno == ENXIO) {
    return MX_NO_DRIVER;
  } else if (errno == EACCES) {
    return MX_NO_PERM;
  } else if (errno == EBUSY) {
    return MX_BUSY;
  } else {
    return MX_NO_RESOURCES;
  }

#elif MX_OS_WINNT
  char buff[80];

  if (priv) {
    strcpy(buff, "\\\\.\\mxctlp");
    *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			 OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (*handle != INVALID_HANDLE_VALUE) {
      return MX_SUCCESS;
    }
    else {
      strcpy(buff, "\\\\.\\mxctl");
      *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			   OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
      if (*handle != INVALID_HANDLE_VALUE) {
	return MX_SUCCESS;
      }
      else {
	return MX_NO_RESOURCES;
      }
    }
  }
  else {
    strcpy(buff, "\\\\.\\mxctl");
    *handle = CreateFile(buff, FILE_READ_ATTRIBUTES | SYNCHRONIZE, 0, NULL,
			 OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (*handle != INVALID_HANDLE_VALUE) {
      return MX_SUCCESS;
    }
    else {
      return MX_NO_RESOURCES;
    }
  }
#elif MX_OS_UDRV
  return mx__udrv_open(0, -1, handle);
#else
#error
#endif
}

MX_FUNC(int)
mx__close(mx_endpt_handle_t handle)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS
  return close(handle) == 0 ? MX_SUCCESS : MX_BAD_BAD_BAD;
#elif MX_OS_WINNT
  /* TODO: Error handling. */
  CloseHandle(handle);
  return 0;
#elif MX_OS_UDRV
  shutdown(handle, SHUT_RDWR);
  return close(handle) == 0 ? MX_SUCCESS : MX_BAD_BAD_BAD;
#else
#error
#endif
}

/*
 * Read a byte from each page of mmaped kernel memory so as
 * to ensure that it has been faulted into the address space.
 * Not doing so leads to page faults the first time each page
 * of mmaped memory is accessed on some OSes (like MacOSX & FreeBSD).
 * These faults can lead to strange artifacts small message
 * benchmarks. 
 */

static void
mx__prefault_mapped_kmem(char *addr, uint32_t len)
{
  volatile char access;
  uint32_t i;

  for (i = 0; i < len; i += 4096) {
    access = addr[i];
  }
}

int
mx__map(struct mx_endpoint *ep)
{
  mx_get_copyblock_t *desc = &ep->desc;
  ep->sendq = mx__mmap(0, desc->sendq_len, ep->handle, desc->sendq_offset, 0);
  if (ep->sendq == MX_MAP_FAILED) {
    goto abort;
  }

  ep->eventq = mx__mmap(0, desc->eventq_len, ep->handle, desc->eventq_offset, 0);
  if (ep->eventq == MX_MAP_FAILED) {
    goto abort;
  }

  mx__prefault_mapped_kmem((char *)ep->eventq, desc->eventq_len);

  ep->recvq = mx__mmap(0, desc->recvq_len, ep->handle, desc->recvq_offset, 0);
  if (ep->recvq == MX_MAP_FAILED) {
    goto abort;
  }

  mx__prefault_mapped_kmem((char *)ep->recvq, desc->recvq_len);

  ep->sram = mx__mmap(0, desc->user_mmapped_sram_len, ep->handle,
		      desc->user_mmapped_sram_offset, 0);
  if (ep->sram == MX_MAP_FAILED) {
    goto abort;
  }
  if (desc->user_mmapped_zreq_len) {
    ep->ze_req = mx__mmap(0, desc->user_mmapped_zreq_len, ep->handle,
			  desc->user_mmapped_zreq_offset, 0);
    if (ep->ze_req == MX_MAP_FAILED) {
      goto abort;
    }
  }
  if (desc->kernel_window_len) {
    ep->kernel_window = mx__mmap(0, desc->kernel_window_len, ep->handle,
			  desc->kernel_window_offset, 1);
    if (ep->kernel_window == MX_MAP_FAILED) {
      goto abort;
    }
  }
  return 0;

 abort:
  mx__unmap(ep);
  return MX_BAD_BAD_BAD;
}

void
mx__unmap(struct mx_endpoint *ep)
{
  mx_get_copyblock_t *desc = &ep->desc;
  if (ep->sram && ep->sram != MX_MAP_FAILED)
    mx__munmap(ep->sram, desc->user_mmapped_sram_len);
  if (ep->recvq && ep->recvq != MX_MAP_FAILED)
    mx__munmap(ep->recvq, ep->desc.recvq_len);
  if (ep->eventq && ep->eventq != MX_MAP_FAILED)
    mx__munmap(ep->eventq, ep->desc.eventq_len);
  if (ep->sendq && ep->sendq != MX_MAP_FAILED)
    mx__munmap(ep->sendq, desc->sendq_len);
  if (ep->ze_req && ep->ze_req != MX_MAP_FAILED)
    mx__munmap(ep->ze_req, ep->desc.user_mmapped_zreq_len);
  if (ep->kernel_window && ep->kernel_window != MX_MAP_FAILED)
    mx__munmap(ep->kernel_window, ep->desc.kernel_window_len);
}

int
mx__ioctl(mx_endpt_handle_t handle, int cmd, void *buf, size_t bufsize)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS
  int ret;
  MX_VALGRIND_PRE_IOCTL_CHECK(cmd, buf);
  ret = ioctl(handle, cmd, buf);
  if (ret == 0) {
    MX_VALGRIND_POST_IOCTL_CHECK(cmd, buf);
    return MX_SUCCESS;
  } else {
    switch (errno) {
    case EBUSY:
      return MX_BUSY;
    case ENOSPC:
      return MX_NO_RESOURCES;
    default:
      return MX_BAD_BAD_BAD;
    }
  }
#elif MX_OS_WINNT
  DWORD bytes;
  DWORD last_error;

  errno = 0;
  if (DeviceIoControl(handle, cmd, buf, (DWORD)bufsize, buf,
		      (DWORD)bufsize, &bytes, NULL) == 0) {
    last_error = GetLastError();
    /* TODO: Magic number. */
    last_error &= ~((1 << 29) | (1 << 30) | (1 << 31));
    errno = last_error;
    /* TODO: Roll this up with other arch. */
    switch (last_error) {
    case EBUSY:
      return MX_BUSY;
    case ENOSPC:
      return MX_NO_RESOURCES;
    default:
      return MX_BAD_BAD_BAD;
    }
  }
  else {
    return MX_SUCCESS;
  }
#elif MX_OS_UDRV
  return mx__udrv_ioctl(handle, cmd, buf, bufsize);
#else
#error
#endif
}

void *
mx__mmap(void *start, size_t length, mx_endpt_handle_t handle, ptrdiff_t offset, int read_only)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_SOLARIS
  void *ptr;
  unsigned flags = PROT_READ;
  if (!read_only)
    flags |= PROT_WRITE;

  return mmap(start, length, flags, MAP_SHARED, handle, offset);
#elif MX_OS_MACOSX || MX_OS_WINNT
  mx_mmap_t x;
  int status;

  x.offset = (uint32_t)offset;
  x.len = (uint32_t)length;
  x.va = (uint64_t)(uintptr_t)start;
  x.requested_permissions = -1;  /* XXX does windows need this?? */
  status =  mx__ioctl(handle, MX_MMAP, &x, sizeof(x));
  if (!status)
    return (void *)(uintptr_t)x.va;
  else {
    return (void *) -1;
  }
#elif MX_OS_UDRV
  mx_always_assert(!read_only);
  return mx__udrv_mmap(start, length, handle, offset);
#else
#error
#endif
}

int
mx__munmap(void *start, size_t length)
{
#if MX_OS_LINUX || MX_OS_FREEBSD || MX_OS_SOLARIS | MX_OS_UDRV
  return munmap(start, length) == 0 ? MX_SUCCESS : MX_BAD_BAD_BAD;
#elif MX_OS_MACOSX || MX_OS_WINNT
  return 0;
#else
#error what OS !?
#endif
}
